////////////////////////////////////////////////////////////////////////////////
//
// © 1999 – 2023 OneSpan North America Inc. All rights reserved.
// All rights reserved. https://www.onespan.com/
//
////////////////////////////////////////////////////////////////////////////////
#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

/// Provides a way to define how different apps of your own can access a fingerprint.
/// You can create your own object conforming to this protocol or use one of the following models:
/// <code>OrchestrationAppPrivate</code>, <code>OrchestrationAppGroup</code> and <code>OrchestrationKeychainSharingGroup</code>
/// <blockquote>
/// Important: Once a fingerprint is created with a certain <code>MSSOrchestration/OrchestrationAccessGroup</code>, it is impossible to modify the access group for that fingerprint.
///
/// </blockquote>
/// <blockquote>
/// Important: Using the same Salt with different access groups, will result in different fingerprints. It means that you can share the fingerprint with only one group of your Apps.
///
/// </blockquote>
/// <blockquote>
/// Tip: Each one of your Apps can be part of many groups. Create the same group in all of your Apps interested to access the shared fingerprint and use that group with this SDK.
///
/// </blockquote>
NS_REFINED_FOR_SWIFT
@protocol OrchestrationAccessGroup <NSObject>

@end

/// Accessible only in the App.
/// Any fingerprint generated in this <code>OrchestrationAccessGroup</code> is only available to the App Sandbox.
NS_REFINED_FOR_SWIFT
@interface OrchestrationAppPrivate : NSObject <OrchestrationAccessGroup>

/// Creates an instance of <code>OrchestrationAppPrivate</code> model conforming to <code>OrchestrationAccessGroup</code> protocol.
/// \param teamId The key for this value is <em>DEVELOPMENT_TEAM</em>. You can find this value by going to <em>Build Settings > Signing > Development Team</em> (ex: PS861PAMJ9.) You can also find this value by logging into Apple developer website.
///
/// \param bundleId The key for this value is <em>PRODUCT_BUNDLE_IDENTIFIER</em>. You can find this value in the General tap of you App target.
///
- (nonnull instancetype)initWithTeamId:(NSString * _Nonnull)teamId bundleId:(NSString * _Nonnull)bundleId;

- (nonnull instancetype)init NS_UNAVAILABLE;
+ (nonnull instancetype)new NS_UNAVAILABLE;
@end

/// Accessible in the App and shared through Keychain sharing groups.
/// Any fingerprint generated in this <code>OrchestrationAccessGroup</code> is  available to other Apps created by your team and being part of the same keychain sharing group.
/// <blockquote>
/// Important: In order to use this access group, the <em>Keychain Sharing</em> capability must be added to your App.
///
/// </blockquote>
NS_REFINED_FOR_SWIFT
@interface OrchestrationKeychainSharingGroup : NSObject <OrchestrationAccessGroup>

/// Creates an instance of <code>OrchestrationKeychainSharingGroup</code> model conforming to <code>OrchestrationAccessGroup</code> protocol.
/// \param teamId The key for this value is <em>DEVELOPMENT_TEAM</em>. You can find this value by going to <em>Build Settings > Signing > Development Team</em> (ex: PS861PAMJ9.) You can also find this value by logging into Apple developer website.
///
/// \param keychainSharingGroup One of the groups available in Keychain groups. You can find the list of available groups or create a new one by going to <em>Signing & Capabilities > Keychain Sharing</em> of your app target.
///
- (nonnull instancetype)initWithTeamId:(NSString * _Nonnull)teamId keychainSharingGroup:(NSString * _Nonnull)keychainSharingGroup;

- (nonnull instancetype)init NS_UNAVAILABLE;
+ (nonnull instancetype)new NS_UNAVAILABLE;
@end

/// Accessible in the App and shared through  App Group containers.
/// Any fingerprint generated in this <code>OrchestrationAccessGroup</code> is  available to any App being part of the same App group container.
/// <blockquote>
/// Important: In order to use this access group, the <em>App Groups</em> capability must be added to your App.
///
/// </blockquote>
NS_REFINED_FOR_SWIFT
@interface OrchestrationAppGroup : NSObject <OrchestrationAccessGroup>

/// Creates an instance of <code>OrchestrationAppGroup</code> model conforming to <code>OrchestrationAccessGroup</code> protocol.
/// \param appGroup One of the groups available in App groups. You can find the list of available groups by going to <em>Signing & Capabilities > App Groups</em> of your app target. You can also find this value in the <em>entitlements</em> file of your App target.
///
- (nonnull instancetype)initWithAppGroup:(NSString * _Nonnull)appGroup;

- (nonnull instancetype)init NS_UNAVAILABLE;
+ (nonnull instancetype)new NS_UNAVAILABLE;
@end

NS_ASSUME_NONNULL_END
